#if UNITY_EDITOR
using System.IO;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using UnityEngine.SceneManagement;

public class SaveSceneAsLevel {
    static StreamWriter writer;
    static Dictionary<SceneLevel, int> levelIDs;
    static int numLevels;
    static bool hub;

    public static void Save(string filename) {
        // Initialize stuff
        hub = SceneManager.GetActiveScene().name == "hub";
        levelIDs = new Dictionary<SceneLevel, int>();
        numLevels = 0;

        var rootBlock = GameObject.Find("RootLevel");
        if (rootBlock == null) {
            Debug.LogWarning("Can't save txt, no root block.");
            return;
        }
        var lp = GameObject.Find("LevelProperties");
        if (lp == null) {
            Debug.LogWarning("Can't save txt, no LevelProperties.");
            return;
        }
        var b = lp.GetComponent<LevelProperties>();

        writer = new StreamWriter(filename, false);

        // Write header
        Write("version 4");
        Write("\n");

        if (b.AttemptOrder != "push,enter,eat,possess" && b.AttemptOrder != "") {
            Write("attempt_order");
            WriteStrArg(b.AttemptOrder);
            Write("\n");
        }

        if (b.ShedEnabled) {
            Write("shed");
            WriteBoolArg(b.ShedEnabled);
            Write("\n");
        }

        if (b.InnerPushEnabled) {
            Write("inner_push");
            WriteBoolArg(b.InnerPushEnabled);
            Write("\n");
        }

        if (b.DrawStyle != LevelProperties.DS.normal) {
            Write("draw_style");
            WriteStrArg(b.DrawStyle.ToString());
            Write("\n");
        }

        if (b.CustomLevelMusic != -1) {
            Write("custom_level_music");
            WriteIntArg(b.CustomLevelMusic);
            Write("\n");
        }

        if (b.CustomLevelPalette != -1) {
            Write("custom_level_palette");
            WriteIntArg(b.CustomLevelPalette);
            Write("\n");
        }

        Write("#");
        Write("\n");

        // Assign level IDs before serializing all the blocks
        AssignLevelIDs(rootBlock);

        // Recursively write contents
        WriteBlock(rootBlock.GetComponent<SceneBlock>(), 0);

        writer.Close();
    }

    static void AssignLevelIDs(GameObject obj) {
        var level = obj.GetComponent<SceneLevel>();
        int id = numLevels;
        levelIDs[level] = id;
        numLevels++;

        foreach (Transform child in obj.transform) {
            var childBlock = child.GetComponent<SceneBlock>();
            if (childBlock != null && child.GetComponent<SceneLevel>() != null) {
                AssignLevelIDs(childBlock.gameObject);
            }
        }
    }

    static void WriteBlock(SceneBlock block, int depth) {
        var level = block.GetComponent<SceneLevel>();
        int id = levelIDs[level];

        WriteTabs(depth);
        Write("Block");
        WriteXY(block.gameObject);

        // Level properties
        WriteIntArg(id);
        WriteIntArg(level.width);
        WriteIntArg(level.height);
        WriteFloatArg(level.hue);
        WriteFloatArg(level.sat);
        WriteFloatArg(level.val);
        WriteFloatArg(level.CamZoomFactor);
        WriteBoolArg(level.FillWithWalls);

        // Block properties
        WriteBoolArg(block.IsPlayer);
        WriteBoolArg(block.Possessable);
        WriteIntArg(block.PlayerOrder);
        WriteBoolArg(block.FlipH);

        // Other block properties
        WriteBoolArg(block.FloatInSpace);
        WriteIntArg(block.SpecialEffect);

        Write("\n");

        var transforms = new List<Transform>();

        // Refs
        transforms.Clear();
        foreach (Transform child in block.transform) {
            var refBlock = child.GetComponent<RefBlock>();
            if (refBlock != null) {
                transforms.Add(child);
            }
        }
        foreach (var t in transforms.ToArray().OrderBy(t => t.localPosition.x*1000 + t.localPosition.y)) {
            WriteRefBlock(t.GetComponent<RefBlock>(), depth + 1);
        }

        // Walls
        transforms.Clear();
        foreach (Transform child in block.transform) {
            var childBlock = child.GetComponent<SceneBlock>();
            var childLevel = child.GetComponent<SceneLevel>();
            if (childBlock != null && childLevel == null) {
                transforms.Add(child);
            }
        }
        foreach (var t in transforms.ToArray().OrderBy(t => t.localPosition.x*1000 + t.localPosition.y)) {
            WriteWall(t.GetComponent<SceneBlock>(), depth + 1);
        }

        // Floors
        transforms.Clear();
        foreach (Transform child in block.transform) {
            var floor = child.GetComponent<SceneFloor>();
            if (floor != null) {
                transforms.Add(child);
            }
        }
        foreach (var t in transforms.ToArray().OrderBy(t => t.localPosition.x*1000 + t.localPosition.y)) {
            WriteFloor(t.GetComponent<SceneFloor>(), depth + 1);
        }

        // Blocks (recursive)
        transforms.Clear();
        foreach (Transform child in block.transform) {
            var childBlock = child.GetComponent<SceneBlock>();
            var childLevel = child.GetComponent<SceneLevel>();
            if (childBlock != null && childLevel != null) {
                transforms.Add(child);
            }
        }
        foreach (var t in transforms.ToArray().OrderBy(t => t.localPosition.x*1000 + t.localPosition.y)) {
            WriteBlock(t.GetComponent<SceneBlock>(), depth + 1);
        }
    }

    static void WriteFloor(SceneFloor floor, int depth) {
        string typeStr = null;
        if (floor.type == SceneFloor.FloorType.Button) typeStr = "Button";
        else if (floor.type == SceneFloor.FloorType.PlayerButton) typeStr = "PlayerButton";
        else if (floor.type == SceneFloor.FloorType.LevelSelectLevel) typeStr = "Portal";
        else if (floor.type == SceneFloor.FloorType.Info) typeStr = "Info";
        else if (floor.type == SceneFloor.FloorType.BreakFloor) typeStr = "Break";
        else if (floor.type == SceneFloor.FloorType.FastTravel) typeStr = "FastTravel";
        else if (floor.type == SceneFloor.FloorType.GalleryFloor) typeStr = "Gallery";
        else if (floor.type == SceneFloor.FloorType.DemoEnd) typeStr = "DemoEnd";
        else if (floor.type == SceneFloor.FloorType.Label) return;
        else {
            Debug.LogWarning("Unrecognized floor type: "+floor.type.ToString());
            return;
        }

        WriteTabs(depth);
        Write("Floor");
        WriteXY(floor.gameObject);
        WriteStrArg(typeStr);

        // Additional properties of portals
        if (floor.type == SceneFloor.FloorType.LevelSelectLevel) {
            string sceneName = floor.SceneName;
            if (sceneName == "") {
                sceneName = "_";
                Debug.LogWarning("scene name is empty string");
            }
            WriteStrArg(sceneName);
            //WriteStrArg(floor.referenceName.Replace(" ", "_"));
        }

        // Additional properties of Infos
        if (floor.type == SceneFloor.FloorType.Info
                || floor.type == SceneFloor.FloorType.Label) {
            string info = floor.info.Replace(" ", "_");
            WriteStrArg(info);
        }

        Write("\n");
    }

    static void WriteRefBlock(RefBlock refBlock, int depth) {
        if (refBlock.SubSceneLevel == null) {
            Debug.LogWarning("RefBlock has null SubSceneLevel. name=" + refBlock.name);
            return;
        }

        if (!levelIDs.ContainsKey(refBlock.SubSceneLevel)) {
            Debug.LogWarning("levelIDs missing key for this ref block level; skipping this ref block.");
            return;
        }

        int refID = levelIDs[refBlock.SubSceneLevel];
        int infEnterID = -1;
        if (refBlock.IsInfEnterBlock) {
            infEnterID = levelIDs[refBlock.InfEnterLevel];
        }

        WriteTabs(depth);
        Write("Ref");
        WriteXY(refBlock.gameObject);

        // ID
        WriteIntArg(refID);

        // Inf
        WriteBoolArg(refBlock.IsExitBlock);
        WriteBoolArg(refBlock.IsInfExitBlock);
        WriteIntArg(refBlock.InfExitNum);
        WriteBoolArg(refBlock.IsInfEnterBlock);
        WriteIntArg(refBlock.InfEnterNum);
        WriteIntArg(infEnterID);

        // Block properties
        WriteBoolArg(refBlock.IsPlayer);
        WriteBoolArg(refBlock.Possessable);
        WriteIntArg(refBlock.PlayerOrder);
        WriteBoolArg(refBlock.FlipH);

        // Other block properties
        WriteBoolArg(refBlock.FloatInSpace);
        WriteIntArg(refBlock.SpecialEffect);

        // Hub
        if (InHub()) {
            WriteStrArg(refBlock.HubShortcutAreaName);
        }

        Write("\n");
    }

    static void WriteWall(SceneBlock wall, int depth) {

        WriteTabs(depth);
        Write("Wall");
        WriteXY(wall.gameObject);

        WriteBoolArg(wall.IsPlayer);
        WriteBoolArg(wall.Possessable);
        WriteIntArg(wall.PlayerOrder);

        if (hub) {
            string us = wall.unlockerScene;
            if (us == "") us = "_";
            WriteStrArg(us);
        }

        Write("\n");
    }

    static void WriteTabs(int depth) {
        for (var i = 0; i < depth; i++) {
            Write("\t");
        }
    }

    static void WriteXY(GameObject obj) {
        if (obj.transform.parent == null) {
            WriteIntArg(-1);
            WriteIntArg(-1);
            return;
        }

        var outerLevel = obj.transform.parent.GetComponent<SceneLevel>();
        int x = outerLevel.LevelX(obj.transform.localPosition.x);
        int y = outerLevel.LevelY(obj.transform.localPosition.y);
        WriteIntArg(x);
        WriteIntArg(y);
    }

    static void WriteStrArg(string arg) {
        Write(" " + arg);
    }

    static void WriteIntArg(int arg) {
        Write(" " + arg.ToString());
    }

    static void WriteFloatArg(float arg) {
        Write(" " + arg.ToString());
    }

    static void WriteBoolArg(bool arg) {
        int n = 0;
        if (arg) n = 1;
        Write(" " + n.ToString());
    }

    static void Write(string str) {
        writer.Write(str);
    }

    static bool InHub() {
        return SceneManager.GetActiveScene().name == "hub";
    }
}
#endif
